; ------------------------ MOTROT.ASM -----------------------------------
; random number generators by Agner Fog 1999
; 32-bit mode version for 80x86 and compatible microprocessors.
; Works with Window 95 and later, and possibly with other 32-bit systems.
;
; The code combines the RANROT type W and the Mother-of-All generators.
; This gives the best possible randomness for the most demanding
; applications. The resolution is 63 bits for XRandom.
;
; XRandomInit must be called before the first call to any of the random functions.
; XRandom returns a floating point number between 0 and 1.
; XIRandom returns an integer in the interval defined by min and max.
; XBRandom returns random bits.
;
; C++ prototypes:
; extern "C" void XRandomInit (int seed);
; extern "C" double XRandom (void);
; extern "C" int XIRandom (int min, int max);
; extern "C" unsigned XBRandom ();
;
;  1999, 2004
; Agner Fog. GNU General Public License www.gnu.org/copyleft/gpl.html
; -----------------------------------------------------------------------

ifdef	??version      ; for Borland TASM v. 3.0 or later
  .386
  include p4macros.asi
  .model flat
else                   ; for Microsoft ML version 6.15 or later
  .686
  .xmm    ; for alignment only
  .model flat
endif


; define appropriate function for error-exit:
; (depends on system):
; ErrorExit equ <FatalAppExitA>

; define parameters for RANROT
JJ      equ     10            ; lag 1
KK      equ     17            ; lag 2, size of circular buffer
R1      equ     19            ; rotate count
R2      equ     27            ; rotate count
SAFE    equ      1            ; define this if you want self-test

; data segment
IFDEF	??version      ; for Borland TASM v. 3.0 or later
  _DATA1 SEGMENT PARA PUBLIC 'DATA'
ELSE
  .DATA
ENDIF
align   16
randp1  DT      1.5           ; used for conversion to float
        DW      0             ; alignment
p1      DD      0             ; pointer in circular buffer
p2      DD      0             ; pointer in circular buffer
randbuf DD      (2*KK) dup(?) ; circular buffer for RANROT
M0      DD      0             ; history buffer for Mother-of-All
M1      DD      0
M2      DD      0
M3      DD      0
MC      DD      0

MF3     DD      2111111111    ; factors for Mother-of-All
MF2     DD      1492
MF1     DD      1776
MF0     DD      5115


IFDEF   SAFE
randbufcopy DD  (4*KK) dup(?) ; double copy of randbuf initial state
IFDEF   ErrorExit
RandMessage1    DB 'Random number generator not initialized',0
RandMessage2    DB 'Random number generator returned to initial state',0
ENDIF
ENDIF

PublicAlias MACRO MangledName ; macro for giving a function alias public names
        MangledName label near
        public MangledName
ENDM

IFDEF	??version      ; Borland TASM
  _DATA1 ENDS
  _TEXT segment DWORD PUBLIC 'CODE'
  FLAT GROUP _DATA1,_TEXT
  ASSUME DS:FLAT,CS:FLAT
ELSE
  .CODE
ENDIF

; code segment

XBRandom PROC NEAR
public XBRandom                ; mangled names are not needed if extern "C" declatation
PublicAlias _XBRandom          ; extern "C" name
;PublicAlias ?XBRandom@@YAIXZ   ; MS mangled name
;PublicAlias @XBRandom$qv       ; Borland mangled name
;PublicAlias _XBRandom__Fv      ; Gnu mangled name (Windows)
;PublicAlias XBRandom__Fv       ; Gnu mangled name (UNIX)
;PublicAlias _Z8XBRandomv       ; Gnu mangled name (UNIX)

; RANROT-W algorithm:
        PUSH    EBX
        MOV     EAX, [p1]       ; ring buffer pointers
        MOV     EDX, [p2]
        MOV     EBX, [randbuf][EAX]
        MOV     ECX, [randbuf][EAX+4]
IFDEF   SAFE    ; include self-test    
        CMP     EBX, [randbufcopy][0] ; self-test first dword of initial state
        JE      R50             ; jump if further compare needed
ENDIF
R20:    ROL     EBX, R1                  ; rotate bits
        ROL     ECX, R2
        PUSH    ESI
        PUSH    EDI
        ADD     EBX, [randbuf][EDX]      ; add two dwords
        ADD     ECX, [randbuf][EDX+4]
        MOV     [randbuf][EAX], ECX      ; save in swapped order
        MOV     [randbuf][EAX+4], EBX
        SUB     EAX, 8                   ; decrement p1
        JNC     SHORT R30
        MOV     EAX, (KK-1)*8            ; wrap around p1
R30:    SUB     EDX, 8                   ; decrement p2
        JNC     SHORT R40
        MOV     EDX, (KK-1)*8            ; wrap around p2
R40:    MOV     [p1], EAX                ; save updated pointers
        MOV     [p2], EDX
; result of RANROT-W is in EBX:ECX

; Mother-of-All algorithm:
        MOV     EAX, [MF3]
        MUL     [M3]             ; X[n-4]
        MOV     ESI,EAX
        MOV     EAX, [M2]        ; X[n-3]
        MOV     EDI,EDX
        MOV     [M3],EAX
        MUL     [MF2]
        ADD     ESI,EAX
        MOV     EAX, [M1]        ; X[n-2]
        ADC     EDI,EDX
        MOV     [M2],EAX
        MUL     [MF1]
        ADD     ESI,EAX
        MOV     EAX,[M0]         ; X[n-1]
        ADC     EDI,EDX
        MOV     [M1],EAX
        MUL     [MF0]
        ADD     EAX,ESI
        ADC     EDX,EDI
        ADD     EAX,[MC]
        ADC     EDX,0
        MOV     [M0],EAX
        MOV     [MC],EDX
; combine output from RANROT and Mother-of-All
        ROL     EDX,8
        ADD     EAX,ECX
        ADD     EDX,EBX
        POP     EDI
        POP     ESI
        POP     EBX
        RET

IFDEF   SAFE
R50:    ; first dword is identical. compare entire buffer
        PUSHAD
        MOV     ESI, OFFSET DS:randbuf
        MOV     EDI, OFFSET DS:randbufcopy+KK*8
        SUB     EDI, EAX                  ; buffer is rotated
        MOV     ECX, KK*2
        CLD
        REPE    CMPSD                     ; compare buffers
        JE      SHORT R60
        POPAD
        JMP     R20                       ; buffers are different. continue
R60:    ; error found. initial state is repeated
IFDEF   ErrorExit
        PUSH    OFFSET DS:RandMessage1
        MOV     EAX, [p1]
        CMP     EAX, [p2]
        JE      SHORT R70
        PUSH    OFFSET DS:RandMessage2
R70:    PUSH    0
        CALL    ErrorExit   ; name of abort procedure depends on system
        EXTRN   ErrorExit:NEAR
ELSE
        ; ErrorExit function not defined. Make division by 0 error
        DIV     ECX
ENDIF
ENDIF
XBRandom ENDP


XRandom PROC NEAR                        ; make random long double float
public XRandom
PublicAlias _XRandom           ; extern "C" name
;PublicAlias ?XRandom@@YANXZ    ; MS mangled name
;PublicAlias @XRandom$qv        ; Borland mangled name
;PublicAlias _XRandom__Fv       ; Gnu mangled name (Windows)
;PublicAlias XRandom__Fv        ; Gnu mangled name (UNIX)
;PublicAlias _Z7XRandomv        ; Gnu mangled name (UNIX)

        CALL    XBRandom                 ; 64 bit random number in EDX:EAX
        OR      EDX, 80000000H           ; fast convert to float
        MOV     DWORD PTR [randp1],EAX
        MOV     DWORD PTR [randp1+4],EDX
        FLD1        
        FLD     TBYTE PTR [randp1]
        FSUBR
        RET
XRandom ENDP


XIRandom PROC   NEAR
public XIRandom
PublicAlias _XIRandom             ; extern "C" name
;PublicAlias ?XIRandom@@YAHHH@Z    ; MS mangled name
;PublicAlias @XIRandom$qii         ; Borland mangled name
;PublicAlias _XIRandom__Fii        ; Gnu mangled name (Windows)
;PublicAlias XIRandom__Fii         ; Gnu mangled name (UNIX)
;PublicAlias _Z8XIRandomii         ; Gnu mangled name (UNIX)

        CALL    XBRandom             ; make random 64 bits
        PUSH    EBX
        MOV     EBX, [ESP+8+4]       ; max
        SUB     EBX, [ESP+4+4]       ; min
        JS      SHORT RERROR         ; max < min
        INC     EBX                  ; max - min + 1
        MOV     ECX, EDX             ; high bits of random number
        MUL     EBX                  ; multiply low 32 bits
        MOV     EAX, ECX
        MOV     ECX, EDX
        MUL     EBX                  ; multiply high 32 bits
        ADD     EAX, ECX
        ADC     EDX, [ESP+4+4]       ; add min
        MOV     EAX, EDX
        POP     EBX
        RET                          ; RET 8 if not _cdecl calling

RERROR: MOV     EAX, 80000000H       ; error exit   
        POP     EBX
        RET                          ; RET 8 if not _cdecl calling
XIRandom ENDP


XRandomInit PROC NEAR
public XRandomInit
PublicAlias _XRandomInit            ; extern "C" name
;PublicAlias ?XRandomInit@@YAXH@Z    ; MS mangled name
;PublicAlias @XRandomInit$qi         ; Borland mangled name
;PublicAlias _XRandomInit__Fi        ; Gnu mangled name (Windows)
;PublicAlias XRandomInit__Fi         ; Gnu mangled name (UNIX)
;PublicAlias _Z11XRandomIniti         ; Gnu mangled name (UNIX)

        MOV     EAX, [ESP+4]    ; seed
        XOR     ECX, ECX
        ; make random numbers and put them into buffers
R80:    IMUL    EAX, 2891336453
        INC     EAX
        MOV     [randbuf][ECX*4], EAX
        INC     ECX
        CMP     ECX, KK*2 + 5
        JB      R80
        FLD1
        FSTP    [randp1]        ; initialize bit 64-79
        MOV     [p1], 0         ; initialize buffer pointers
        MOV     [p2], JJ*8
IFDEF   SAFE
        PUSHAD                  ; make 2 copies of initial state
        MOV     ECX, KK*2
        MOV     ESI, OFFSET DS:randbuf
        MOV     EDI, OFFSET DS:randbufcopy
        PUSH    ECX
        PUSH    ESI
        CLD
        REP     MOVSD
        POP     ESI
        POP     ECX
        REP     MOVSD
        INC     [randbufcopy][0]   ; avoid error message
        CALL    XBRandom           ; prepare first random number
        DEC     [randbufcopy][0]
        POPAD
ELSE
        CALL    XBRandom           ; prepare first random number
ENDIF
        PUSH    EDI
        MOV     EDI, 30
R90:    CALL    XBRandom
        DEC     EDI
        JNZ     R90
        POP     EDI                                                
        RET     0                  ; RET 4 if not _cdecl calling
XRandomInit ENDP

IFDEF	??version      ; Borland TASM
  _TEXT ENDS
ENDIF


; Example of calling from C++ program:

; #include "arandom.h"
; int i;
; int seed = time(0);
; XRandomInit (seed);
; for (i=0; i<20; i++) {
;   printf ("\n%14.10f  %2i", XRandom(), XIRandom(0,99));}

        END
