/*
	Copyright 2006 Johannes Kopf (kopf@inf.uni-konstanz.de)

	Implementation of the algorithms described in the paper

	Recursive Wang Tiles for Real-Time Blue Noise
	Johannes Kopf, Daniel Cohen-Or, Oliver Deussen, Dani Lischinski
	In ACM Transactions on Graphics 25, 3 (Proc. SIGGRAPH 2006)

	If you use this software for research purposes, please cite
	the aforementioned paper in any resulting publication.
	
	You can find updated versions and other supplementary materials
	on our homepage:
	http://graphics.uni-konstanz.de/publications/2006/blue_noise
*/

#include <math.h>
#include "points.h"
#include "randoma.h"

extern int numPoints, numTiles, numColors, numSubtiles, numSubdivs;
extern Tile * tiles, * sources;
ubyte colors[];

void saveTileSet(const char * fileName)
{
	FILE * fout = fopen(fileName, "wb");
	
	fwritei(fout, numTiles);
	fwritei(fout, numSubtiles);
	fwritei(fout, numSubdivs);

	for (int t = 0; t < numTiles; t++)
	{
		fwritei(fout, tiles[t].n);
		fwritei(fout, tiles[t].e);
		fwritei(fout, tiles[t].s);
		fwritei(fout, tiles[t].w);

		for (int i = 0; i < numSubdivs; i++)
			for (int j = 0; j < sqri(numSubtiles); j++)
				fwritei(fout, tiles[t].subdivs[i][j]);

		fwritei(fout, (int)tiles[t].points.size());
		for (int i = 0; i < (int)tiles[t].points.size(); i++)
		{
			fwritef(fout, tiles[t].points[i].x);
			fwritef(fout, tiles[t].points[i].y);
			fwritei(fout, tiles[t].points[i].n);
			fwritei(fout, tiles[t].points[i].cr);
			fwritei(fout, tiles[t].points[i].cg);
			fwritei(fout, tiles[t].points[i].cb);
		}

		fwritei(fout, (int)tiles[t].subPoints.size());
		for (int i = 0; i < (int)tiles[t].subPoints.size(); i++)
		{
			fwritef(fout, tiles[t].subPoints[i].x);
			fwritef(fout, tiles[t].subPoints[i].y);
			fwritei(fout, tiles[t].subPoints[i].n);
			fwritei(fout, tiles[t].subPoints[i].cr);
			fwritei(fout, tiles[t].subPoints[i].cg);
			fwritei(fout, tiles[t].subPoints[i].cb);
		}
	}

	fclose(fout);
}

void loadTileSet(const char * fileName)
{
	FILE * fin = fopen(fileName, "rb");
	
	numTiles = freadi(fin);
	numSubtiles = freadi(fin);
	numSubdivs = freadi(fin);

	tiles = new Tile[numTiles];

	for (int i = 0; i < numTiles; i++)
	{
		tiles[i].n = freadi(fin);
		tiles[i].e = freadi(fin);
		tiles[i].s = freadi(fin);
		tiles[i].w = freadi(fin);

		for (int j = 0; j < numSubdivs; j++)
		{
			std::vector<int> subdiv;
			for (int k = 0; k< sqri(numSubtiles); k++)
				subdiv.push_back(freadi(fin));
			tiles[i].subdivs.push_back(subdiv);
		}

		int numPoints = freadi(fin);
		for (int j = 0; j < numPoints; j++)
		{
			Point p;
			p.x = freadf(fin);
			p.y = freadf(fin);
			p.n = freadi(fin);
			p.cr = freadi(fin);
			p.cg = freadi(fin);
			p.cb = freadi(fin);
			tiles[i].points.push_back(p);
		}

		int numSubPoints = freadi(fin);
		for (int j = 0; j < numSubPoints; j++)
		{
			Point p;
			p.x = freadf(fin);
			p.y = freadf(fin);
			p.n = freadi(fin);
			p.cr = freadi(fin);
			p.cg = freadi(fin);
			p.cb = freadi(fin);
			tiles[i].subPoints.push_back(p);
		}
	}

	fclose(fin);
}

// ************************************************************************
// ************************************************************************
// ********                                                        ********
// ********    functions for step 1: creating progressive tiles    ********
// ********                                                        ********
// ************************************************************************
// ************************************************************************

// blue noise dart throwing algorithm from
// MCCOOL, M., AND FIUME, E. 1992.
// Hierarchical poisson disk sampling distributions.
// In Proc. Graphics interface 92, 94105.

std::vector<Point> toroidalDartThrowing(int numPoints, ubyte cr = 0, ubyte cg = 0, ubyte cb = 0)
{
	float dartRadius = 0.5f;
	std::vector<Point> points;

	// set the first point in the upper left edge.
	// this alleviates almost all problems with the tile corners.
	points.push_back(Point(0, 0, 0, 128, 128, 128));

	// generate the remaining points
	printf("    ");
	for (int k = 1; k < numPoints; k++)
	{
		printf("\b\b\b\b%3d%%", k*100/numPoints);

		int misses = 0;
		while(true)
		{
			Point p((float)TRandom(), (float)TRandom(), k, cr, cg, cb);

			bool miss = false;
			for (int j = 0; j < (int)points.size(); j++)
			{
				// toroidal distance
				float distX = fabsf(p.x - points[j].x);
				float distY = fabsf(p.y - points[j].y);
				if (distX > 0.5f)
					distX = 1.f - distX;
				if (distY > 0.5f)
					distY = 1.f - distY;

				// is a previous point inside the dart radius?
				if (sqrf(distX)+sqrf(distY) < sqrf(dartRadius))
				{
					miss = true;
					break;
				}
			}

			if (!miss)
			{
				// success
				points.push_back(p);
				break;
			}
			else
			{
				// after some number of consecutive failures
				if (++misses > 1000)
				{
					// reduce the dart radius
					misses = 0;
					dartRadius *= 0.99f;
				}
			}
		}
	}
	printf("\b\b\b\b");
	return points;
}

// extract points inside a closed polygon (and append to target)

void extractPointsInPoly(std::vector<Vec2> & poly, std::vector<Point> & source, std::vector<Point> & target)
{
	for (int p = 0; p < (int)source.size(); p++)
	{
		int npol = (int)poly.size();
		int i, j, c = 0;
		for (i = 0, j = npol-1; i < npol; j = i++) {
			if ((((poly[i].y <= source[p].y) && (source[p].y < poly[j].y)) ||
				((poly[j].y <= source[p].y) && (source[p].y < poly[i].y))) &&
				(source[p].x < (poly[j].x - poly[i].x) * (source[p].y - poly[i].y) / (poly[j].y - poly[i].y) + poly[i].x))
			c = !c;
		}

		if (c == 1)
			target.push_back(source[p]);
	}
}

struct Edge
{
	Edge() {};
	Edge(int _p, int _q, int _tl, int _tr) : p(_p), q(_q), tl(_tl), tr(_tr) {};
	int p, q;
	float cost;
	int tl, tr;
};

std::vector<Vec2> verts;		// voronoi vertices
std::vector<Edge> edges;		// voronoi edges
std::vector<Point> overlay;		// overlay of source tiles

// overlay two source tiles

void calcOverlay(std::vector<Point> & source1, std::vector<Point> & source2)
{
	overlay.clear();
	for (int i = 0; i < (int)source1.size(); i++)
		overlay.push_back(source1[i]);
	for (int i = 0; i < (int)source2.size(); i++)
		overlay.push_back(source2[i]);
}

// beware! extremely hacky code to compute the voronoi diagram.
// this could very likely be done much more reliable and faster
// don't try to understand this ;-)

struct VoronoiEdge
{
	VoronoiEdge(Vec2 _p, Vec2 _q, int _tl, int _tr) : p(_p), q(_q), tl(_tl), tr(_tr) {};

	Vec2 p;
	Vec2 q;
	int tl, tr;
};

void intersectRegion(std::vector<VoronoiEdge> & voronoi, Vec3 & interSection, int tl, int tr)
{
	bool closeFirst = false;

	for (int i = 0; i < (int)voronoi.size(); i++)
	{
		// calc sides of current voronoi edge i wrt. intersection line
		float sp = Vec3(voronoi[i].p)*interSection;
		float sq = Vec3(voronoi[i].q)*interSection;

		if ((sp < 0) && (sq < 0))
		{
			// edge complete outside -> cut it
			voronoi.erase(voronoi.begin() + i);
			i--;
			if (i == (int)voronoi.size()-1)
				closeFirst = true;
		}
		else if ((sp >= 0) && (sq >= 0))
		{
			// edge complete inside
			// do nothing :-)
		}
		else if ((sp >= 0) && (sq < 0))
		{
			// p inside but q outside
			// set q to intersection point
			Vec2 dif = voronoi[i].q-voronoi[i].p;
			float t = (Vec3(voronoi[i].p)*(-interSection))/(dif.x*interSection.x+dif.y*interSection.y);
			voronoi[i].q = voronoi[i].p+dif*t;
			if (i == (int)voronoi.size()-1)
				closeFirst = true;
		}
		else if ((sp < 0) && (sq >= 0))
		{
			// p outside but q inside
			// set p to intersection point
			Vec2 dif = voronoi[i].q-voronoi[i].p;
			float t = (Vec3(voronoi[i].p)*(-interSection))/(dif.x*interSection.x+dif.y*interSection.y);
			voronoi[i].p = voronoi[i].p+dif*t;
			// insert new edge closing the voronoi region
			if (i == 0)
				closeFirst = true;
			else
			{
				VoronoiEdge newEdge(voronoi[i-1].q, voronoi[i].p, tl, tr);
				voronoi.insert(voronoi.begin()+i, newEdge);
				i++;
			}
		}
		else
			error("voronoi calculation error (this should never happen)");
	}

	if (closeFirst)
	{
		VoronoiEdge newEdge(voronoi[(int)voronoi.size()-1].q, voronoi[0].p, tl, tr);
		voronoi.push_back( newEdge );
	}
}

void calcVoronoi()
{
	printf("[voronoi]     ");
	
	verts.clear();
	edges.clear();

    // compute the voronoi cells for each point individually
	for (int i = 0; i < (int)overlay.size(); i++)
	{
		printf("\b\b\b\b%3d%%", i*100/(int)overlay.size());

		// initial voronoi cell spans the whole tile
		std::vector<VoronoiEdge> cell;
		cell.push_back( VoronoiEdge(Vec2(0,0), Vec2(1,0), -1, i) );
		cell.push_back( VoronoiEdge(Vec2(1,0), Vec2(1,1), -1, i) );
		cell.push_back( VoronoiEdge(Vec2(1,1), Vec2(0,1), -1, i) );
		cell.push_back( VoronoiEdge(Vec2(0,1), Vec2(0,0), -1, i) );

		// intersect cell with other points
		for (int j = 0; j < (int)overlay.size(); j++)
		{
			// don't intersect point with itself!
			if ((overlay[j].x == overlay[i].x) && (overlay[j].y == overlay[i].y))
				continue;

			// calc perpendicular bisector
			Vec2 normal(overlay[j].x-overlay[i].x, overlay[j].y-overlay[i].y);
			normal.normalize();
			Vec2 midPoint(overlay[j].x+overlay[i].x, overlay[j].y+overlay[i].y);
			midPoint /= 2.f;
			Vec3 interSection(-normal.x, -normal.y, normal*midPoint);
			intersectRegion(cell, interSection, j, i);
		}

		// update verts and edges structures
		for (int i = 0; i < (int)cell.size(); i++)
		{
			int vp = -1, vq = -1;
			// try to find p in verts
			for (int j = 0; j < (int)verts.size(); j++)
			{
				if ((fabs(verts[j].x - cell[i].p.x) < 0.001f) && (fabs(verts[j].y - cell[i].p.y) < 0.001f))
					vp = j;
			}
			if (vp == -1)
			{
				verts.push_back( Vec2(cell[i].p.x, cell[i].p.y) );
				vp = (int)verts.size()-1;
			}
			// try to find q in verts
			for (int j = 0; j < (int)verts.size(); j++)
			{
				if ((fabs(verts[j].x - cell[i].q.x) < 0.001f) && (fabs(verts[j].y - cell[i].q.y) < 0.001f))
					vq = j;
			}
			if (vq == -1)
			{
				verts.push_back( Vec2(cell[i].q.x, cell[i].q.y) );
				vq = (int)verts.size()-1;
			}

			edges.push_back( Edge(vp, vq, cell[i].tl, cell[i].tr) );
			edges.push_back( Edge(vq, vp, cell[i].tr, cell[i].tl) );
		}
	}

	printf("\b\b\b\b\b");

	//printf("%d vertices, %d edges.\n", (int)verts.size(), (int)edges.size());
}

// assigns a cost to each edge as described in the paper.

void calcCosts()
{
	printf("[costs]");
	
	// init costs to dist
	float maxDist = FLOAT_MIN;
	for (int i = 0; i < (int)edges.size(); i++)
	{
		if ((edges[i].tl == -1) || (edges[i].tr == -1))
		{
			edges[i].cost = 0;
			continue;
		}
		Point & p1 = overlay[edges[i].tl];
		Point & p2 = overlay[edges[i].tr];
		float dist = sqrtf(sqrf(p1.x-p2.x)+sqrf(p1.y-p2.y));
		edges[i].cost = dist;
		maxDist = maxf(maxDist, dist);
	}

	// calc final costs (inversed normalized cost to the power of 10)
	for (int i = 0; i < (int)edges.size(); i++)
		edges[i].cost = powf( (maxDist - edges[i].cost)/(maxDist/4), 10 );
}

// calculates the shortest path using dijkstras algorithm

std::vector<Vec2> calcBestCut(Vec2 _vs, Vec2 _ve)
{
	printf("[cut]");

	int vs = -1, ve = -1;

	for (int i = 0; i < (int)verts.size(); i++)
	{
		if ((fabsf(verts[i].x-_vs.x) < 0.001f) && (fabsf(verts[i].y-_vs.y) < 0.001f))
			vs = i;
		if ((fabsf(verts[i].x-_ve.x) < 0.001f) && (fabsf(verts[i].y-_ve.y) < 0.001f))
			ve = i;
	}

	if ((vs == -1) || (ve == -1))
		error("start or end vertex not found...");

	int numVerts = (int)verts.size();
	float * d = new float[numVerts];
	int * previous = new int[numVerts];
	std::vector<int> S, Q;

	for (int v = 0; v < numVerts; v++)
	{
		d[v] = FLOAT_MAX;
		previous[v] = -1;
		Q.push_back(v);
	}
	d[vs] = 0;

	while(Q.size() > 0)
	{
		// find shortest path so far
		int found = 0;
		float dval = FLOAT_MAX;
		for (int i = 0; i < (int)Q.size(); i++)
		{
			if (d[Q[i]] < dval)
			{
				dval = d[Q[i]];
				found = i;
			}
		}
		int shortest = Q[found];

		// shortest path found?
		if (shortest == ve)
			break;

		S.push_back(shortest);

		Q.erase(Q.begin() + found);

		for (int e = 0; e < (int)edges.size(); e++)
		{
			if (edges[e].p != shortest)
				continue;

			if (d[edges[e].q] > d[shortest]+edges[e].cost)
			{
				d[edges[e].q] = d[shortest]+edges[e].cost;
				previous[edges[e].q] = shortest;
			}
		}
	}

	std::vector<Vec2> cut;
	{
	int v = ve;
	do
	{
		cut.push_back(verts[v]);
		v = previous[v];
	} while(v != vs);
	}
	cut.push_back(verts[vs]);

    delete[] d;
	delete[] previous;

	return cut;
}

// *******************
// create source tiles
void createSourceTiles()
{
	printf("creating source tiles...\n");
	for (int i = 0; i < numColors; i++)
	{
		printf("creating source tile %d/%d ", i+1, numColors);
		sources[i].points = toroidalDartThrowing(numPoints, colors[i*3+0], colors[i*3+1], colors[i*3+2]);
		printf("[done]\n");
	}
}

// *******************************************************
// create individual body parts for each of the wang tiles
void createWangBodies()
{
	printf("\ncreating wang tile body point sets...\n");
	for (int i = 0; i < numTiles; i++)
	{
		printf("creating body for tile %d/%d ", i+1, numTiles);
		tiles[i].points = toroidalDartThrowing(numPoints);
		printf("[done]\n");
	}
}

// *****************
// merge north edges
void mergeNorth()
{
	printf("\nmerging north edges...\n");
	for (int i = 0; i < numTiles; i++)
	{
		printf("merging north edge for tile %d/%d ", i+1, numTiles);
		
		calcOverlay(sources[tiles[i].n].points, tiles[i].points);
		calcVoronoi();
		calcCosts();
		std::vector<Vec2> cut = calcBestCut( Vec2(0, 0), Vec2(1, 0) );

		std::vector<Point> points;
		std::vector<Vec2> poly = cut;
		poly.push_back( Vec2(0, 0) );
		extractPointsInPoly(poly, sources[tiles[i].n].points, points);
		poly = cut;
		poly.push_back( Vec2(0, 1) );
		poly.push_back( Vec2(1, 1) );
		extractPointsInPoly(poly, tiles[i].points, points);

		tiles[i].points = points;
		printf("[done]\n");
	}
}

// ****************
// merge east edges
void mergeEast()
{
	printf("\nmerging east edges...\n");
	for (int i = 0; i < numTiles; i++)
	{
		printf("merging east edge for tile %d/%d ", i+1, numTiles);

		calcOverlay(sources[tiles[i].e].points, tiles[i].points);
		calcVoronoi();
		calcCosts();
		std::vector<Vec2> cut = calcBestCut( Vec2(1, 0), Vec2(1, 1) );

		std::vector<Point> points;
		std::vector<Vec2> poly = cut;
		poly.push_back( Vec2(1, 0) );
		extractPointsInPoly(poly, sources[tiles[i].e].points, points);
		poly = cut;
		poly.push_back( Vec2(0, 0) );
		poly.push_back( Vec2(0, 1) );
		extractPointsInPoly(poly, tiles[i].points, points);

		tiles[i].points = points;
		printf("[done]\n");
	}
}

// *****************
// merge south edges
void mergeSouth()
{
	printf("\nmerging south edges...\n");
	for (int i = 0; i < numTiles; i++)
	{
		printf("merging south edge for tile %d/%d ", i+1, numTiles);

		calcOverlay(sources[tiles[i].s].points, tiles[i].points);
		calcVoronoi();
		calcCosts();
		std::vector<Vec2> cut = calcBestCut( Vec2(1, 1), Vec2(0, 1) );

		std::vector<Point> points;
		std::vector<Vec2> poly = cut;
		poly.push_back( Vec2(1, 1) );
		extractPointsInPoly(poly, sources[tiles[i].s].points, points);
		poly = cut;
		poly.push_back( Vec2(1, 0) );
		poly.push_back( Vec2(0, 0) );
		extractPointsInPoly(poly, tiles[i].points, points);

		tiles[i].points = points;
		printf("[done]\n");
	}
}

// ****************
// merge west edges
void mergeWest()
{
	printf("\nmerging west edges...\n");
	for (int i = 0; i < numTiles; i++)
	{
		printf("merging west edge for tile %d/%d ", i+1, numTiles);

		calcOverlay(sources[tiles[i].w].points, tiles[i].points);
		calcVoronoi();
		calcCosts();
		std::vector<Vec2> cut = calcBestCut( Vec2(0, 1), Vec2(0, 0) );

		std::vector<Point> points;
		std::vector<Vec2> poly = cut;
		poly.push_back( Vec2(0, 1) );
		extractPointsInPoly(poly, sources[tiles[i].w].points, points);
		poly = cut;
		poly.push_back( Vec2(1, 1) );
		poly.push_back( Vec2(1, 0) );
		extractPointsInPoly(poly, tiles[i].points, points);

		tiles[i].points = points;
		printf("[done]\n");
	}
}

// *************************
// fix progression sequence
void fixRanking()
{
	printf("\nfixing progression sequences... ");
	for (int tn = 0; tn < numTiles; tn++)
	{
		Tile & t = tiles[tn];

		// interleaving points
		for (int i = 1; i < (int)t.points.size()-1; i++)
		{
			for (int j = i+1; j < (int)t.points.size(); j++)
			{
				if (t.points[i].n > t.points[j].n)
					std::swap(t.points[i], t.points[j]);
			}
		}

		// fixing too close points
		for (int i = 1; i < (int)t.points.size()-1; i++)
		{
			for (int k = i; k < (int)t.points.size(); k++)
			{
				// find next point that is not too close
				float minDist = FLOAT_MAX;
				for (int j = 0; j < i; j++)
					minDist = minf(minDist, (Vec2(t.points[k].x,t.points[k].y)-Vec2(t.points[j].x,t.points[j].y)).magnitude());

				if (minDist > 0.5f*1.f/sqrtf(i+1.f))
				{
					std::swap(t.points[i], t.points[k]);
					break;
				}
			}
		}
	}
	printf("[done]\n");
}

// ***************************************************************************
// ***************************************************************************
// ********                                                           ********
// ********    functions for step 2: making the tile set recursive    ********
// ********                                                           ********
// ***************************************************************************
// ***************************************************************************

std::vector< std::vector<int> > borders;

void initSubdivision()
{
	printf("init subdivision... ");

	// create edge color-sequences
	borders.clear();

	// create first set of edge-sequences (north/south)
	for (int i = 0; i < numColors/2; i++)
	{
		std::vector<int> seq;
		for (int j = 0; j < numSubtiles; j++)
		{
			int c;
			bool duplicate;
			do
			{
				c = TIRandom(0, numColors/2-1);
				duplicate = false;
				for (int k = 0; k < i; k++)
					if (c == borders[k][j])
						duplicate = true;
			} while(duplicate);
			seq.push_back(c);
		}
		borders.push_back(seq);
	}

	// create second set of colors (east west)
	for (int i = numColors/2; i < numColors; i++)
	{
		std::vector<int> seq;
		for (int j = 0; j < numSubtiles; j++)
		{
			int c;
			bool duplicate;
			do
			{
				c = TIRandom(numColors/2, numColors-1);
				duplicate = false;
				for (int k = numColors/2; k < i; k++)
					if (c == borders[k][j])
						duplicate = true;
			} while(duplicate);
			seq.push_back(c);
		}
		borders.push_back(seq);
	}

	for (int i = 0; i < numTiles; i++)
	{
		tiles[i].subdivs.clear();

		for (int j = 0; j < numSubdivs; j++)
		{
			std::vector<int> subdiv;
			for (int k = 0; k < sqri(numSubtiles); k++)
				subdiv.push_back( TIRandom(0, numTiles-1) );
			tiles[i].subdivs.push_back(subdiv);
		}
	}

	printf("[done]\n");
}

int countErrors(Tile & t, int n)
{
	std::vector<int> & subdiv = t.subdivs[n];

	int numErrors = 0;
	for (int y = 0; y < numSubtiles; y++)
	{
		for (int x = 0; x < numSubtiles; x++)
		{
			int cn, ce, cs, cw;		// edge constraints
			cn = (y > 0 ? tiles[subdiv[x+(y-1)*numSubtiles]].s : borders[t.n][x]);
			ce = (x < numSubtiles-1 ? tiles[subdiv[(x+1)+y*numSubtiles]].w : borders[t.e][y]);
			cs = (y < numSubtiles-1 ? tiles[subdiv[x+(y+1)*numSubtiles]].n : borders[t.s][x]);
			cw = (x > 0 ? tiles[subdiv[(x-1)+y*numSubtiles]].e : borders[t.w][y]);

			// check constraints, count faults
			if (tiles[subdiv[x+y*numSubtiles]].n != cn)
				numErrors++;
			if (tiles[subdiv[x+y*numSubtiles]].e != ce)
				numErrors++;
			if (tiles[subdiv[x+y*numSubtiles]].s != cs)
				numErrors++;
			if (tiles[subdiv[x+y*numSubtiles]].w != cw)
				numErrors++;
		}
	}

	return numErrors;
}

int countErrors()
{
	int numErrors = 0;
	for (int i = 0; i < numTiles; i++)
		for (int j = 0; j < numSubdivs; j++)
			numErrors += countErrors(tiles[i], j);
	return numErrors;
}

void repairSubdivision(int numSteps)
{
	int matches[65536];

	for (int tn = 0; tn < numTiles; tn++)
	{
		Tile & t = tiles[tn];

		for (int n = 0; n < numSubdivs; n++)
		{
			std::vector<int> & subdiv = t.subdivs[n];

			int numErrors = countErrors(t, n);
			if (numErrors == 0)
				continue;

			int misses = 0;
			for (int step = 0; step < numSteps; step++)
			{
				int x = TIRandom(0, numSubtiles-1);
				int y = TIRandom(0, numSubtiles-1);

				// check edge constraints
				int cn, ce, cs, cw, numErrors = 0;
				cn = (y > 0 ? tiles[subdiv[x+(y-1)*numSubtiles]].s : borders[t.n][x]);
				if (tiles[subdiv[x+y*numSubtiles]].n != cn)
					numErrors++;
				ce = (x < numSubtiles-1 ? tiles[subdiv[(x+1)+y*numSubtiles]].w : borders[t.e][y]);
				if (tiles[subdiv[x+y*numSubtiles]].e != ce)
					numErrors++;
				cs = (y < numSubtiles-1 ? tiles[subdiv[x+(y+1)*numSubtiles]].n : borders[t.s][x]);
				if (tiles[subdiv[x+y*numSubtiles]].s != cs)
					numErrors++;
				cw = (x > 0 ? tiles[subdiv[(x-1)+y*numSubtiles]].e : borders[t.w][y]);
				if (tiles[subdiv[x+y*numSubtiles]].w != cw)
					numErrors++;

				if (numErrors == 0)
				{
					// found a valid tile
					misses++;
					if (misses > 1000)
					{
						if (countErrors(t, n) == 0)
							break;
						misses = 0;
					}
				}

				int numMatches = 0;
				for (int j = 0; j < numTiles; j++)
				{
					// count errors
					int checkErrors = 0;
					if (tiles[j].n != cn)
						checkErrors++;
					if (tiles[j].e != ce)
						checkErrors++;
					if (tiles[j].s != cs)
						checkErrors++;
					if (tiles[j].w != cw)
						checkErrors++;

					// only consider tiles that have fewer or equal errors
					if (checkErrors > numErrors)
						continue;

					// to increase the "randomness" we penelize having the same tile next to itself
					int rep = 1;
					if (x > 0)
						if (tiles[subdiv[y*numSubtiles+(x-1)]].w != tiles[j].w)
							rep += 25;
					if (x < numSubtiles-1)
						if (tiles[subdiv[y*numSubtiles+(x+1)]].e != tiles[j].e)
							rep += 25;
					if (y > 0)
						if (tiles[subdiv[(y-1)*numSubtiles+x]].n != tiles[j].n)
							rep += 25;
					if (y < numSubtiles-1)
						if (tiles[subdiv[(y+1)*numSubtiles+x]].s != tiles[j].s)
							rep += 25;

					for (int k = 0; k < rep; k++)
					{
						matches[numMatches] = j;
						numMatches++;
					}
				}

				if (numMatches > 0)
					subdiv[y*numSubtiles+x] = matches[TIRandom(0, numMatches-1)];
			}
		}
	}
}

void createSubdivision()
{
	initSubdivision();

	printf("repairing subdivision...          errors");
	int numSteps = 1000, stepCount = 0, stepLimit = sqri(numSubtiles)*20000;
	int numErrors;
	while((numErrors = countErrors()) > 0)
	{
		printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%8d errors", numErrors);
		repairSubdivision(numSteps);

		stepCount += numSteps;
		if (stepCount > stepLimit)
		{
			stepCount = 0;
			printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b[failed]             \n");
			initSubdivision();
		}
	}
	printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b[done]             \n");
}

// *******************************************************************************************
// *******************************************************************************************
// ********                                                                           ********
// ********    functions for step 3: making the tile set progressive and recursive    ********
// ********                                                                           ********
// *******************************************************************************************
// *******************************************************************************************

int numCandidates;

void relaxPoints()
{
	printf("relaxing points... error: 9.99999999");
	float movement;
	do
	{
		// init relax vectors
		for (int i = 0; i < numTiles; i++)
		{
			if (tiles[i].d.size() != tiles[i].points.size())
			{
				tiles[i].d.resize(tiles[i].points.size());
				for (int j = 0; j < (int)tiles[i].d.size(); j++)
					tiles[i].d[j] = Vec2(0,0);
			}
		}

		// calc relax vectors
		float scale = 1.f/numSubtiles;
		for (int i = 0; i < numTiles; i++)
		{
			for (int j = 0; j < (int)tiles[i].points.size(); j++)
			{
				float minDist = FLOAT_MAX;
				for (int y = 0; y < numSubtiles; y++)
				{
					for (int x = 0; x < numSubtiles; x++)
					{
						Tile & sub = tiles[tiles[i].subdivs[0][x+y*numSubtiles]];
						for (int k = 0; k < (int)sub.points.size(); k++)
						{
							Vec2 p((x+sub.points[k].x)*scale, (y+sub.points[k].y)*scale);
							float dist = sqrf(tiles[i].points[j].x - p.x) + sqrf(tiles[i].points[j].y - p.y);
							if (dist < minDist)
							{
								minDist = dist;
								tiles[i].d[j] = p;
							}
						}
					}
				}
			}
		}

		// move points
		movement = 0;
		for (int i = 0; i < numTiles; i++)
		{
			Tile & t = tiles[i];
			for (int i = 0; i < (int)t.points.size(); i++)
			{
				movement += (sqrf(t.points[i].x-t.d[i].x) + sqrf(t.points[i].y-t.d[i].y)) * 0.1f;
				t.points[i].x += (t.d[i].x-t.points[i].x) * 0.1f;
				t.points[i].y += (t.d[i].y-t.points[i].y) * 0.1f;
			}
		}
		printf("\b\b\b\b\b\b\b\b\b\b%1.8f", movement);
	} while (movement > 0.0000001f);
	printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b[done]            \n");
}

void initFinalRanking()
{
	printf("init final ranking stage... ");

	numCandidates = 0;

	// init
	for (int i = 0; i < numTiles; i++)
	{
		Tile & t = tiles[i];

		// ******************************************************************
		// Create a 3x3 wang tiling "neighborhood" for each tile. In the dart
		// throwing stage we check also for dart collisions with these neighbors.
		// This avoids artifacts at tile borders almost completely.

		// random neighborhood
		for (int j = 0; j < 3; j++)
			for (int k = 0; k < 3; k++)
				t.neighbors[j][k] = TIRandom(0, numTiles-1);
		t.neighbors[1][1] = i;

		// fix N,E,S,W neighbors until they fit with center tile
		while(tiles[t.neighbors[1][0]].s != t.n)
			t.neighbors[1][0] = TIRandom(0, numTiles-1);
		while(tiles[t.neighbors[2][1]].w != t.e)
			t.neighbors[2][1] = TIRandom(0, numTiles-1);
		while(tiles[t.neighbors[1][2]].n != t.s)
			t.neighbors[1][2] = TIRandom(0, numTiles-1);
		while(tiles[t.neighbors[0][1]].e != t.w)
			t.neighbors[0][1] = TIRandom(0, numTiles-1);

        // fix NW,NE,SE,SW (diagonal) neighbors	until they fix with previous tiles
		while((tiles[t.neighbors[0][0]].e != tiles[t.neighbors[1][0]].w) && (tiles[t.neighbors[0][0]].s != tiles[t.neighbors[0][1]].n))
			t.neighbors[0][0] = TIRandom(0, numTiles-1);
		while((tiles[t.neighbors[2][0]].w != tiles[t.neighbors[1][0]].e) && (tiles[t.neighbors[2][0]].s != tiles[t.neighbors[2][1]].n))
			t.neighbors[2][0] = TIRandom(0, numTiles-1);
		while((tiles[t.neighbors[2][2]].w != tiles[t.neighbors[1][2]].e) && (tiles[t.neighbors[2][2]].n != tiles[t.neighbors[2][1]].s))
			t.neighbors[2][2] = TIRandom(0, numTiles-1);
		while((tiles[t.neighbors[0][2]].e != tiles[t.neighbors[1][2]].w) && (tiles[t.neighbors[0][2]].n != tiles[t.neighbors[0][1]].s))
			t.neighbors[0][2] = TIRandom(0, numTiles-1);

		// ******************************************************************
		// populate candidate vectors with all sub points that DO NOT coincide
		// with base points. The sub points that DO coincide with base points
		// are thrown away.
		t.candidates.clear();
		t.check = t.points;

		for (int y = 0; y < numSubtiles; y++)
		{
			for (int x = 0; x < numSubtiles; x++)
			{
				Tile & st = tiles[ t.subdivs[0][y*numSubtiles+x] ];
				
				for (int j = 0; j < (int)st.points.size(); j++)
				{
					Point p((x+st.points[j].x)/numSubtiles, (y+st.points[j].y)/numSubtiles);
					bool found = false;
					for (int k = 0; k < (int)t.check.size(); k++)
					{
						if ((fabsf(p.x-t.check[k].x) < 0.0001f) && (fabsf(p.y-t.check[k].y) < 0.0001f))
						{
							// this sub point coincides with one of the relaxed base points
							found = true;
							break;
						}
					}

					// consider only sub points which do not coincide with a base point
					if (!found)
						t.candidates.push_back(p);
				}
			}
		}

		numCandidates += (int)t.candidates.size();
		t.dartRadius = 0.1f;
		t.misses = 0;
	}

	printf("[done]\n");
}

void createFinalRanking()
{
	printf("final ranking... candidates left: %8d", numCandidates);
	while(numCandidates > 0)
	{
		printf("\b\b\b\b\b\b\b\b%08d", numCandidates);

		// go through all tiles and try throwing a dart on each
		for (int i = 0; i < numTiles; i++)
		{
			Tile & t = tiles[i];

			if (t.candidates.empty())
				continue;

			while(true)
			{
				// select a random candidate
				int tp = TIRandom(0, (int)t.candidates.size()-1);

				// check with all previous placed darts (also in neighbor tiles)
				bool miss = false;
				for (int y = 0; y < 3; y++)
				{
					for (int x = 0; x < 3; x++)
					{
						Tile & tc = tiles[t.neighbors[x][y]];
						for (int j = 0; j < (int)tc.check.size(); j++)
						{
							// compute distance
							float distX = fabsf(t.candidates[tp].x - (tc.check[j].x + (x-1)));
							float distY = fabsf(t.candidates[tp].y - (tc.check[j].y + (y-1)));
							if (sqrf(distX)+sqrf(distY) < sqrf(t.dartRadius))
							{
								// collision
								miss = true;
								break;
							}
						}
						if (miss)
							break;
					}
					if (miss)
						break;
				}

				if (!miss)
				{
					// point is good. remove from candidates and append to subpoints
					t.subPoints.push_back(t.candidates[tp]);
					t.check.push_back(t.candidates[tp]);
					t.candidates.erase(t.candidates.begin() + tp);
					numCandidates--;
					break;
				}
				else
				{
					if (++t.misses > 1000)
					{
						t.misses = 0;
						t.dartRadius *= 0.99f;
					}
				}
			}
		}
	}
	printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b[done]                               \n");
}