/*
	Copyright 2006 Johannes Kopf (kopf@inf.uni-konstanz.de)

	Implementation of the algorithms described in the paper

	Recursive Wang Tiles for Real-Time Blue Noise
	Johannes Kopf, Daniel Cohen-Or, Oliver Deussen, Dani Lischinski
	In ACM Transactions on Graphics 25, 3 (Proc. SIGGRAPH 2006)

	If you use this software for research purposes, please cite
	the aforementioned paper in any resulting publication.
	
	You can find updated versions and other supplementary materials
	on our homepage:
	http://graphics.uni-konstanz.de/publications/2006/blue_noise
*/

#ifndef POINTS_H
#define POINTS_H

#include <vector>
#include "util.h"
#include "geometry.h"

struct Point
{
	Point() {};
	Point(float _x, float _y) : x(_x), y(_y) {};
	Point(float _x, float _y, int _n, ubyte _cr, ubyte _cg, ubyte _cb) : x(_x), y(_y), n(_n), cr(_cr), cg(_cg), cb(_cb) {};

	float x;				// coordinates
	float y;
	int n;					// rank
	ubyte cr, cg, cb;		// color
};

struct Tile
{
	Tile() { };
	Tile(int _n, int _e, int _s, int _w) : n(_n), e(_e), s(_s), w(_w) { };

	int n, e, s, w;

	std::vector< std::vector<int> > subdivs;
	std::vector<Point> points, subPoints;

	// needed for the relaxation
	std::vector<Vec2> d;
	int neighbors[3][3];
	std::vector<Point> candidates, check;
	float dartRadius;
	int misses;
};

void saveTileSet(const char * fileName);
void loadTileSet(const char * fileName);

// functions for step 1: creating progressive tiles
void createSourceTiles();
void createWangBodies();
void mergeNorth();
void mergeEast();
void mergeSouth();
void mergeWest();
void fixRanking();

// functions for step 2: making the tile set recursive
void initSubdivision();
int countErrors();
void repairSubdivision(int numSteps);
void createSubdivision();

// functions for step 3: making the tile set progressive and recursive
void relaxPoints();
void initFinalRanking();
void createFinalRanking();

#endif